/**
 * JavaScript لوحة التحكم
 * Dashboard JavaScript
 */

// المتغيرات العامة
let currentTab = 'home';
let projects = [];
let copiedSlug = null;

// قوالب النماذج
const templates = [
    { id: 'landscaping', name: 'تنسيق حدائق', description: 'تصميم طبيعي وجذاب يعكس جمال الطبيعة، مثالي لشركات تنسيق الحدائق والمشاتل.' },
    { id: 'moving', name: 'نقل عفش', description: 'تصميم يركز على السرعة والأمان، مثالي لشركات نقل الأثاث والخدمات اللوجستية.' },
    { id: 'used-furniture', name: 'شراء اثاث مستعمل', description: 'تصميم بسيط ومباشر يسهل عملية البيع والشراء، مثالي لتجار الأثاث المستعمل.' },
    { id: 'contracting', name: 'مقاولات عامة', description: 'تصميم احترافي يعكس القوة والثقة، مثالي لشركات المقاولات والبناء.' },
    { id: 'recruitment', name: 'استقدام عمالة', description: 'تصميم رسمي وموثوق، مثالي لمكاتب الاستقدام والخدمات العمالية.' },
    { id: 'decor', name: 'ديكور وتصميم داخلي', description: 'تصميم فني وأنيق يبرز جمال الأعمال، مثالي لمهندسي الديكور والمصممين.' },
    { id: 'pest-control', name: 'رش حشرات', description: 'تصميم نظيف يوحي بالتعقيم والحماية، مثالي لشركات مكافحة الحشرات.' }
];

// عند تحميل الصفحة
document.addEventListener('DOMContentLoaded', function() {
    if (IS_LOGGED_IN) {
        loadProjects();
    }
    renderTemplates();
});

// تبديل القائمة المحمولة
function toggleMobileMenu() {
    const sidebar = document.getElementById('sidebar');
    const overlay = document.getElementById('overlay');
    sidebar.classList.toggle('translate-x-full');
    overlay.classList.toggle('hidden');
}

// تغيير التبويب النشط
function setActiveTab(tab) {
    currentTab = tab;
    
    // تحديث العناوين
    const titles = {
        'home': 'لوحة التحكم',
        'projects': 'صفحاتي',
        'templates': 'مكتبة النماذج'
    };
    document.getElementById('page-title').textContent = titles[tab];
    
    // إخفاء جميع الأقسام
    document.querySelectorAll('.content-section').forEach(section => {
        section.classList.add('hidden');
    });
    
    // إظهار القسم المحدد
    document.getElementById(tab + '-section').classList.remove('hidden');
    
    // تحديث التنقل
    document.querySelectorAll('.nav-item').forEach(item => {
        item.classList.remove('bg-blue-50', 'text-blue-600', 'font-bold');
        item.classList.add('text-gray-600', 'hover:bg-gray-50');
    });
    document.getElementById('nav-' + tab).classList.add('bg-blue-50', 'text-blue-600', 'font-bold');
    document.getElementById('nav-' + tab).classList.remove('text-gray-600', 'hover:bg-gray-50');
    
    // إغلاق القائمة المحمولة
    if (window.innerWidth < 768) {
        toggleMobileMenu();
    }
}

// تحميل المشاريع
async function loadProjects() {
    try {
        const response = await fetch(SITE_URL + '/api/projects.php?action=list');
        const result = await response.json();
        
        if (result.success) {
            projects = result.data;
            renderProjects();
            updateStats();
        }
    } catch (error) {
        console.error('Error loading projects:', error);
    }
}

// تحديث الإحصائيات
function updateStats() {
    const totalProjects = projects.length;
    const totalViews = projects.reduce((sum, p) => sum + parseInt(p.views || 0), 0);
    
    document.getElementById('stat-projects').textContent = totalProjects;
    document.getElementById('stat-views').textContent = totalViews;
}

// عرض المشاريع
function renderProjects() {
    const recentList = document.getElementById('recent-projects-list');
    const allList = document.getElementById('all-projects-list');
    
    if (projects.length === 0) {
        const emptyState = `
            <div class="bg-white rounded-xl border border-gray-200 p-12 text-center">
                <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4 text-gray-400">
                    <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 11H5m14 0a2 2 0 012 2v6a2 2 0 01-2 2H5a2 2 0 01-2-2v-6a2 2 0 012-2m14 0V9a2 2 0 00-2-2M5 11V9a2 2 0 012-2m0 0V5a2 2 0 012-2h6a2 2 0 012 2v2M7 7h10"></path>
                    </svg>
                </div>
                <h3 class="text-lg font-bold text-gray-900 mb-2">لا توجد صفحات بعد</h3>
                <p class="text-gray-500 mb-6">أنشئ صفحتك الأولى الآن وابدأ في استقبال العملاء.</p>
                <button onclick="setActiveTab('templates')" class="text-blue-600 font-bold hover:underline">تصفح النماذج</button>
            </div>
        `;
        recentList.innerHTML = emptyState;
        allList.innerHTML = emptyState;
        return;
    }
    
    const tableHtml = `
        <div class="bg-white rounded-xl border border-gray-200 overflow-hidden shadow-sm">
            <div class="overflow-x-auto">
                <table class="w-full text-right min-w-[600px]">
                    <thead class="bg-gray-50 border-b border-gray-200">
                        <tr>
                            <th class="px-6 py-4 text-sm font-semibold text-gray-600">اسم الصفحة</th>
                            <th class="px-6 py-4 text-sm font-semibold text-gray-600">النموذج</th>
                            <th class="px-6 py-4 text-sm font-semibold text-gray-600">تاريخ الإنشاء</th>
                            <th class="px-6 py-4 text-sm font-semibold text-gray-600">المشاهدات</th>
                            <th class="px-6 py-4 text-sm font-semibold text-gray-600">الإجراءات</th>
                        </tr>
                    </thead>
                    <tbody class="divide-y divide-gray-100">
                        ${projects.map(project => {
                            const template = templates.find(t => t.id === project.template_id);
                            return `
                                <tr class="hover:bg-gray-50 transition-colors">
                                    <td class="px-6 py-4 font-medium text-gray-900">${project.name || 'بدون عنوان'}</td>
                                    <td class="px-6 py-4 text-gray-600">${template?.name || project.template_id}</td>
                                    <td class="px-6 py-4 text-gray-500 text-sm">${formatDate(project.created_at)}</td>
                                    <td class="px-6 py-4 text-gray-500">${project.views || 0}</td>
                                    <td class="px-6 py-4">
                                        <div class="flex items-center gap-2">
                                            <a href="${SITE_URL}/p.php?slug=${project.slug}" target="_blank" class="text-blue-600 hover:underline text-sm">عرض</a>
                                            <button onclick="copyLink('${project.slug}')" class="text-gray-400 hover:text-gray-600 p-1 rounded" title="نسخ الرابط">
                                                ${copiedSlug === project.slug ? 
                                                    '<svg class="w-4 h-4 text-green-500" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path></svg>' :
                                                    '<svg class="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z"></path></svg>'
                                                }
                                            </button>
                                            <a href="${SITE_URL}/builder.php?id=${project.id}" class="text-gray-600 hover:text-blue-600 font-medium text-sm mr-2">تعديل</a>
                                        </div>
                                    </td>
                                </tr>
                            `;
                        }).join('')}
                    </tbody>
                </table>
            </div>
        </div>
    `;
    
    recentList.innerHTML = tableHtml;
    allList.innerHTML = tableHtml;
}

// عرض النماذج
function renderTemplates() {
    const grid = document.getElementById('templates-grid');
    grid.innerHTML = templates.map(template => `
        <div class="bg-white rounded-xl border border-gray-200 overflow-hidden hover:shadow-md transition-shadow flex flex-col">
            <div class="h-48 bg-gray-100 relative border-b border-gray-100">
                <div class="absolute inset-0 flex items-center justify-center text-gray-400">
                    <svg class="w-12 h-12" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 5a1 1 0 011-1h14a1 1 0 011 1v2a1 1 0 01-1 1H5a1 1 0 01-1-1V5zM4 13a1 1 0 011-1h6a1 1 0 011 1v6a1 1 0 01-1 1H5a1 1 0 01-1-1v-6zM16 13a1 1 0 011-1h2a1 1 0 011 1v6a1 1 0 01-1 1h-2a1 1 0 01-1-1v-6z"></path>
                    </svg>
                </div>
            </div>
            <div class="p-5 flex flex-col flex-grow">
                <h3 class="text-lg font-bold text-gray-900 mb-2">${template.name}</h3>
                <p class="text-gray-500 text-sm mb-6 line-clamp-2 flex-grow">${template.description}</p>
                <a href="${SITE_URL}/builder.php?template=${template.id}" class="block w-full text-center bg-blue-600 hover:bg-blue-700 text-white font-medium py-2.5 rounded-lg transition-colors">
                    استخدم هذا النموذج
                </a>
            </div>
        </div>
    `).join('');
}

// نسخ الرابط
function copyLink(slug) {
    const url = `${window.location.origin}/p.php?slug=${slug}`;
    navigator.clipboard.writeText(url).then(() => {
        copiedSlug = slug;
        renderProjects();
        setTimeout(() => {
            copiedSlug = null;
            renderProjects();
        }, 2000);
    });
}

// تنسيق التاريخ
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('ar-SA');
}
