<?php
/**
 * إدارة الجلسات
 * Session Management
 */

// بدء الجلسة
function startSession() {
    if (session_status() === PHP_SESSION_NONE) {
        session_name(SESSION_NAME);
        session_start();
        
        // تجديد رمز الجلسة بشكل دوري للأمان
        if (!isset($_SESSION['created'])) {
            $_SESSION['created'] = time();
        } elseif (time() - $_SESSION['created'] > 1800) {
            session_regenerate_id(true);
            $_SESSION['created'] = time();
        }
    }
}

/**
 * تسجيل دخول المستخدم
 * Login user
 */
function loginUser($userId, $remember = false) {
    $_SESSION['user_id'] = $userId;
    $_SESSION['logged_in'] = true;
    $_SESSION['login_time'] = time();
    
    if ($remember) {
        // إنشاء جلسة دائمة في قاعدة البيانات
        $sessionToken = generateToken();
        $expires = date('Y-m-d H:i:s', time() + SESSION_LIFETIME);
        
        insert('sessions', [
            'user_id' => $userId,
            'session_token' => $sessionToken,
            'expires' => $expires
        ]);
        
        // حفظ في كوكيز
        setcookie(SESSION_NAME . '_token', $sessionToken, time() + SESSION_LIFETIME, '/');
    }
    
    // تحديث آخر تسجيل دخول
    update('users', ['updated_at' => date('Y-m-d H:i:s')], 'id = :id', ['id' => $userId]);
}

/**
 * تسجيل خروج المستخدم
 * Logout user
 */
function logoutUser() {
    $userId = $_SESSION['user_id'] ?? null;
    
    if ($userId) {
        // حذف الجلسات من قاعدة البيانات
        delete('sessions', 'user_id = :user_id', ['user_id' => $userId]);
    }
    
    // مسح الكوكيز
    if (isset($_COOKIE[SESSION_NAME . '_token'])) {
        setcookie(SESSION_NAME . '_token', '', time() - 3600, '/');
    }
    
    // تدمير الجلسة
    $_SESSION = [];
    session_destroy();
}

/**
 * التحقق من تسجيل الدخول
 * Check if user is logged in
 */
function isLoggedIn() {
    startSession();
    
    // التحقق من الجلسة العادية
    if (isset($_SESSION['user_id']) && isset($_SESSION['logged_in'])) {
        return true;
    }
    
    // التحقق من الجلسة الدائمة (Remember Me)
    if (isset($_COOKIE[SESSION_NAME . '_token'])) {
        $token = $_COOKIE[SESSION_NAME . '_token'];
        $sql = "SELECT user_id FROM sessions WHERE session_token = :token AND expires > NOW()";
        $session = fetchOne($sql, ['token' => $token]);
        
        if ($session) {
            loginUser($session['user_id'], false);
            return true;
        }
    }
    
    return false;
}

/**
 * الحصول على المستخدم الحالي
 * Get current user
 */
function getCurrentUser() {
    if (!isLoggedIn()) {
        return null;
    }
    
    $userId = $_SESSION['user_id'] ?? null;
    if (!$userId) {
        return null;
    }
    
    $sql = "SELECT id, email, name, image, google_id, created_at FROM users WHERE id = :id";
    return fetchOne($sql, ['id' => $userId]);
}

/**
 * طلب تسجيل الدخول
 * Require login
 */
function requireLogin() {
    if (!isLoggedIn()) {
        if (isAjaxRequest()) {
            error('غير مصرح. يرجى تسجيل الدخول.', 401);
        } else {
            redirect(SITE_URL . '/index.php?login_required=1');
        }
    }
}

/**
 * التحقق من طلب AJAX
 * Check if AJAX request
 */
function isAjaxRequest() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && 
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) === 'xmlhttprequest';
}

/**
 * حفظ أو تحديث حساب OAuth
 * Save or update OAuth account
 */
function saveOAuthAccount($userId, $provider, $providerAccountId, $tokenData) {
    // التحقق إذا كان الحساب موجود
    $sql = "SELECT id FROM oauth_accounts WHERE provider = :provider AND provider_account_id = :provider_account_id";
    $existing = fetchOne($sql, [
        'provider' => $provider,
        'provider_account_id' => $providerAccountId
    ]);
    
    $data = [
        'user_id' => $userId,
        'provider' => $provider,
        'provider_account_id' => $providerAccountId,
        'access_token' => $tokenData['access_token'] ?? null,
        'refresh_token' => $tokenData['refresh_token'] ?? null,
        'expires_at' => $tokenData['expires_at'] ?? null,
        'token_type' => $tokenData['token_type'] ?? null,
        'scope' => $tokenData['scope'] ?? null
    ];
    
    if ($existing) {
        unset($data['provider']);
        unset($data['provider_account_id']);
        update('oauth_accounts', $data, 'id = :id', ['id' => $existing['id']]);
    } else {
        insert('oauth_accounts', $data);
    }
}

/**
 * البحث عن مستخدم بواسطة Google ID
 * Find user by Google ID
 */
function findUserByGoogleId($googleId) {
    $sql = "SELECT * FROM users WHERE google_id = :google_id";
    return fetchOne($sql, ['google_id' => $googleId]);
}

/**
 * البحث عن مستخدم بواسطة البريد الإلكتروني
 * Find user by email
 */
function findUserByEmail($email) {
    $sql = "SELECT * FROM users WHERE email = :email";
    return fetchOne($sql, ['email' => $email]);
}

/**
 * إنشاء مستخدم جديد من Google
 * Create new user from Google
 */
function createUserFromGoogle($googleData) {
    $data = [
        'google_id' => $googleData['id'],
        'email' => $googleData['email'],
        'name' => $googleData['name'],
        'image' => $googleData['picture'] ?? null,
        'email_verified' => ($googleData['verified_email'] ?? false) ? 1 : 0
    ];
    
    return insert('users', $data);
}

// بدء الجلسة تلقائياً
startSession();
