// DOM Elements
const savedSearchesContainer = document.getElementById('savedSearchesContainer');
const noSavedSearchesAlert = document.getElementById('noSavedSearches');

// Event Listeners
document.addEventListener('DOMContentLoaded', () => {
    // التحقق من حالة تسجيل الدخول
    const currentUser = auth.getCurrentUser();
    
    // إذا لم يكن المستخدم مسجل الدخول، قم بتوجيهه إلى صفحة تسجيل الدخول
    if (!currentUser) {
        window.location.href = 'login.html';
        return;
    }
    
    // تحديث واجهة المستخدم بناءً على حالة تسجيل الدخول
    updateUIForAuthStatus(currentUser);
    
    // عرض عمليات البحث المحفوظة
    displaySavedSearches(currentUser.savedSearches || []);
});

/**
 * تحديث واجهة المستخدم بناءً على حالة تسجيل الدخول
 * @param {Object|null} user - بيانات المستخدم المسجل أو null إذا لم يكن هناك مستخدم
 */
function updateUIForAuthStatus(user) {
    const userMenuContainer = document.getElementById('userMenuContainer');
    
    if (!userMenuContainer) return;
    
    if (user) {
        // إذا كان المستخدم مسجل الدخول
        userMenuContainer.innerHTML = `
            <div class="dropdown">
                <button class="btn btn-outline-light dropdown-toggle" type="button" id="userDropdown" data-bs-toggle="dropdown" aria-expanded="false">
                    <i class="bi bi-person-circle"></i> ${user.name}
                </button>
                <ul class="dropdown-menu dropdown-menu-end" aria-labelledby="userDropdown">
                    <li><a class="dropdown-item" href="profile.html"><i class="bi bi-person"></i> الملف الشخصي</a></li>
                    <li><a class="dropdown-item" href="saved-searches.html"><i class="bi bi-bookmark"></i> عمليات البحث المحفوظة</a></li>
                    <li><hr class="dropdown-divider"></li>
                    <li><a class="dropdown-item" href="#" id="logoutBtn"><i class="bi bi-box-arrow-right"></i> تسجيل الخروج</a></li>
                </ul>
            </div>
        `;
        
        // إضافة مستمع حدث لزر تسجيل الخروج
        document.getElementById('logoutBtn').addEventListener('click', auth.handleLogout);
    } else {
        // إذا لم يكن المستخدم مسجل الدخول
        userMenuContainer.innerHTML = `
            <a href="login.html" class="btn btn-outline-light">
                <i class="bi bi-box-arrow-in-right"></i> تسجيل الدخول
            </a>
        `;
    }
}

/**
 * عرض عمليات البحث المحفوظة
 * @param {Array} savedSearches - قائمة عمليات البحث المحفوظة
 */
function displaySavedSearches(savedSearches) {
    if (!savedSearches || savedSearches.length === 0) {
        // إذا لم تكن هناك عمليات بحث محفوظة
        noSavedSearchesAlert.classList.remove('d-none');
        savedSearchesContainer.innerHTML = '';
        return;
    }
    
    // إخفاء رسالة عدم وجود عمليات بحث
    noSavedSearchesAlert.classList.add('d-none');
    
    // فرز عمليات البحث بترتيب زمني تنازلي (الأحدث أولاً)
    const sortedSearches = [...savedSearches].sort((a, b) => {
        return new Date(b.savedAt) - new Date(a.savedAt);
    });
    
    // إنشاء HTML لعرض عمليات البحث
    const searchesHTML = sortedSearches.map(search => {
        const savedDate = new Date(search.savedAt).toLocaleString('ar-SA');
        
        return `
            <div class="card mb-3">
                <div class="card-body">
                    <div class="d-flex justify-content-between align-items-center mb-2">
                        <h5 class="card-title">${search.keyword} في ${search.location}</h5>
                        <span class="badge bg-secondary">${savedDate}</span>
                    </div>
                    <p class="card-text">عدد النتائج: ${search.resultCount}</p>
                    <div class="d-flex gap-2">
                        <button class="btn btn-primary btn-sm run-search" data-id="${search.id}">
                            <i class="bi bi-search"></i> تشغيل البحث
                        </button>
                        <button class="btn btn-danger btn-sm delete-search" data-id="${search.id}">
                            <i class="bi bi-trash"></i> حذف
                        </button>
                    </div>
                </div>
            </div>
        `;
    }).join('');
    
    // إضافة HTML إلى الحاوية
    savedSearchesContainer.innerHTML = searchesHTML;
    
    // إضافة مستمعي الأحداث لأزرار تشغيل البحث والحذف
    document.querySelectorAll('.run-search').forEach(button => {
        button.addEventListener('click', () => runSavedSearch(button.dataset.id));
    });
    
    document.querySelectorAll('.delete-search').forEach(button => {
        button.addEventListener('click', () => deleteSavedSearch(button.dataset.id));
    });
}

/**
 * تشغيل بحث محفوظ
 * @param {string} searchId - معرف البحث المحفوظ
 */
function runSavedSearch(searchId) {
    const currentUser = auth.getCurrentUser();
    
    if (!currentUser || !currentUser.savedSearches) {
        alert('حدث خطأ أثناء تشغيل البحث');
        return;
    }
    
    // البحث عن البحث المحفوظ بواسطة المعرف
    const search = currentUser.savedSearches.find(s => s.id === searchId);
    
    if (!search) {
        alert('لم يتم العثور على البحث المحفوظ');
        return;
    }
    
    // إنشاء معلمات URL للبحث
    const params = new URLSearchParams({
        keyword: search.keyword,
        location: search.location,
        resultCount: search.resultCount
    });
    
    // توجيه المستخدم إلى الصفحة الرئيسية مع معلمات البحث
    window.location.href = `index.html?${params.toString()}`;
}

/**
 * حذف بحث محفوظ
 * @param {string} searchId - معرف البحث المحفوظ
 */
function deleteSavedSearch(searchId) {
    if (!confirm('هل أنت متأكد من رغبتك في حذف هذا البحث المحفوظ؟')) {
        return;
    }
    
    const currentUser = auth.getCurrentUser();
    
    if (!currentUser) {
        alert('يجب تسجيل الدخول لحذف البحث المحفوظ');
        return;
    }
    
    // الحصول على قائمة المستخدمين
    const users = getUsers();
    const userIndex = users.findIndex(u => u.id === currentUser.id);
    
    if (userIndex === -1) {
        alert('حدث خطأ أثناء حذف البحث');
        return;
    }
    
    // حذف البحث من قائمة البحث المحفوظة للمستخدم
    const savedSearches = users[userIndex].savedSearches || [];
    users[userIndex].savedSearches = savedSearches.filter(search => search.id !== searchId);
    
    // حفظ التغييرات
    saveUsers(users);
    
    // تحديث المستخدم الحالي في التخزين
    const storage = localStorage.getItem('currentUser') ? localStorage : sessionStorage;
    currentUser.savedSearches = users[userIndex].savedSearches;
    storage.setItem('currentUser', JSON.stringify(currentUser));
    
    // تحديث واجهة المستخدم
    displaySavedSearches(currentUser.savedSearches);
    
    alert('تم حذف البحث المحفوظ بنجاح');
}

/**
 * الحصول على قائمة المستخدمين
 * @returns {Array} - قائمة المستخدمين
 */
function getUsers() {
    const usersJson = localStorage.getItem('users');
    return usersJson ? JSON.parse(usersJson) : [];
}

/**
 * حفظ قائمة المستخدمين
 * @param {Array} users - قائمة المستخدمين
 */
function saveUsers(users) {
    localStorage.setItem('users', JSON.stringify(users));
}
